<?php

namespace App\Modules\popupForm\Models\spisak;

use \App\Models\BaseModel as BaseModel;

/**
 * Handles SEO metadata for categories.
 * Provides CRUD operations and automatic generation of default SEO data.
 */
class MODELpop__categorySEO extends BaseModel {

    protected $table         = '_category_seo';
    protected $primaryKey    = 'seo_id';
    protected $allowedFields = [
        'category_id', 'seo_title', 'seo_description', 'focus_keyword',
        'canonical_url', 'noindex', 'seo_slug'
    ];
    protected $returnType    = 'array';
    protected $useTimestamps = true;

    /**
     * Fetch SEO data for a specific category.
     */
    public function getByCategoryId(int $categoryId): ?array {
        return $this -> where('category_id', $categoryId) -> first();
    }

    /**
     * Insert or update SEO data for a given category.
     */
    public function saveSEO(array $data): bool {
        if (empty($data['category_id'])) {
            return false;
        }

        $data   = $this -> generateDefaultSEO($data);
        $exists = $this -> where('category_id', $data['category_id']) -> first();

        if ($exists) {
            return $this -> update($exists['seo_id'], $data);
        }

        return $this -> insert($data) !== false;
    }

    public function generateDefaultSEO(array $data): array {
        helper(['text', 'url']);

        $name        = trim($data['category_name'] ?? '');
        $description = trim(strip_tags($data['category_description'] ?? ''));
        $categoryId  = $data['category_id'] ?? 0;

        // Load characteristics
        $chars = $this -> db -> table(self::TBL_CATEGORY_ATTRIBUTE)
                        -> select('value')
                        -> where('category_id', $categoryId)
                        -> get() -> getResultArray();

        $charValues = array_map(fn($c) => trim($c['value']), $chars);
        $charValues = array_filter($charValues);
        $charList   = implode(', ', $charValues);

        // SEO Title
        if (empty($data['seo_title']) && $name) {
            $extra             = $charValues ? " – " . implode(', ', array_slice($charValues, 0, 3)) : '';
            $data['seo_title'] = ucfirst($name) . $extra;
        }

        // SEO Description
        if (empty($data['seo_description'])) {
            if ($description) {
                $data['seo_description'] = rtrim(character_limiter($description, 160), '.') . '.';
            } else {
                $data['seo_description'] = "Открий богато разнообразие от {$name}. Предлагаме " .
                        ($charList ?: "различни модели и варианти") .
                        " на конкурентни цени.";
            }
        }

        // Canonical URL (relative)
        if (empty($data['canonical_url']) && !empty($data['seo_slug'])) {
            $data['canonical_url'] = '/category/' . $data['seo_slug'];
        }

        // Keywords
        if (empty($data['focus_keyword'])) {
            $keywords              = [];
            $keywords              = array_merge($keywords, explode(' ', mb_strtolower($name)));
            $keywords              = array_merge($keywords, $charValues);
            $keywords              = array_unique(array_filter($keywords));
            $data['focus_keyword'] = implode(', ', array_slice($keywords, 0, 8));
        }

        // Slug
        if (empty($data['seo_slug']) && $name) {
            $slug = $this -> transliterateToLatin($name);
            $slug = strtolower(url_title($slug, '-', true));

            $exists = $this -> where('seo_slug', $slug) -> first();
            if ($exists) {
                $slug .= '-' . $categoryId;
            }

            $data['seo_slug'] = $slug ?: 'category-' . $categoryId;
        }

        // Noindex default
        if (!isset($data['noindex'])) {
            $data['noindex'] = 0;
        }

        return $data;
    }

    /**
     * Transliterate Cyrillic → Latin for SEO-friendly slugs.
     */
    private function transliterateToLatin(string $text): string {
        $map = [
            'а' => 'a', 'б' => 'b', 'в' => 'v', 'г' => 'g', 'д' => 'd', 'е' => 'e', 'ж' => 'zh', 'з' => 'z', 'и' => 'i', 'й' => 'y', 'к' => 'k',
            'л' => 'l', 'м' => 'm', 'н' => 'n', 'о' => 'o', 'п' => 'p', 'р' => 'r', 'с' => 's', 'т' => 't', 'у' => 'u', 'ф' => 'f', 'х' => 'h',
            'ц' => 'ts', 'ч' => 'ch', 'ш' => 'sh', 'щ' => 'sht', 'ъ' => 'a', 'ь' => '', 'ю' => 'yu', 'я' => 'ya',
            'А' => 'A', 'Б' => 'B', 'В' => 'V', 'Г' => 'G', 'Д' => 'D', 'Е' => 'E', 'Ж' => 'Zh', 'З' => 'Z', 'И' => 'I', 'Й' => 'Y', 'К' => 'K',
            'Л' => 'L', 'М' => 'M', 'Н' => 'N', 'О' => 'O', 'П' => 'P', 'Р' => 'R', 'С' => 'S', 'Т' => 'T', 'У' => 'U', 'Ф' => 'F', 'Х' => 'H',
            'Ц' => 'Ts', 'Ч' => 'Ch', 'Ш' => 'Sh', 'Щ' => 'Sht', 'Ъ' => 'A', 'Ь' => '', 'Ю' => 'Yu', 'Я' => 'Ya'
        ];

        return strtr($text, $map);
    }

    /**
     * Retrieve a list of SEO records with optional search filter.
     */
    public function getSeoList($search = '') {
        $builder = $this -> db -> table($this -> table);

        if ($search) {
            $builder -> like('seo_title', $search)
                    -> orLike('seo_description', $search)
                    -> orLike('focus_keyword', $search);
        }

        return $builder -> get() -> getResultArray();
    }

    /**
     * Updates a single SEO field or creates a new record if missing.
     * Works for both categories and products depending on table.
     */
    public function updateField($id, $field, $value, $extra = []) {
        // Разрешени полета (Allowed fields)
        if (!in_array($field, [
                    'seo_title', 'seo_description', 'focus_keyword',
                    'canonical_url', 'noindex', 'seo_slug'
                ])) {
            return false;
        }

        $isProduct = str_contains($this -> table, 'product');
        $fkField   = $isProduct ? 'product_id' : 'category_id';
        $fkValue   = $extra[$fkField] ?? null;

        if (empty($fkValue)) {
            return false;
        }

        $record = null;
        if (!empty($id)) {
            $record = $this -> find($id);
        } else {
            $record = $this -> where($fkField, $fkValue) -> first();
        }


        if (!$record) {
            $data     = [
                $fkField => $fkValue,
                $field   => $value,
            ];
            $insertId = $this -> insert($data, true);
            return $insertId ? true : false;
        }

        return $this -> update($record[$this -> primaryKey], [$field => $value]);
    }

}
