<?php

namespace App\Modules\popupForm\Models\product;

use CodeIgniter\Model;

/**
 * Handles SEO metadata management for products.
 *
 * Provides CRUD operations and default value generation for product SEO data,
 * including meta title, description, keywords, canonical URL, slug, index flags,
 * and image metadata.
 *
 * Table: _product_seo
 */
class MODELpop__productSEO extends Model {

    public function generateFreshSEO(array $product): array {
        helper(['text', 'url']);

        $name      = trim($product['product_name'] ?? '');
        $short     = trim(strip_tags($product['short_description'] ?? ''));
        $long      = trim(strip_tags($product['description'] ?? ''));
        $kod       = trim($product['kod'] ?? '');
        $oem       = trim($product['oem'] ?? '');
        $productId = $product['product_id'] ?? 0;

        $data = [];

        /* TITLE */
        $title             = $name;
        if ($kod !== '')
            $title             .= ", КОД: " . $kod;
        if ($oem !== '')
            $title             .= ", OEM: " . $oem;
        $data['seo_title'] = $title;

        /* DESCRIPTION */
        $base = $short ?: $long;
        if ($base) {
            $desc                    = character_limiter($base, 165);
            $data['seo_description'] = rtrim($desc, '.') . '.';
        } else {
            $data['seo_description'] = "Предлагаме " . mb_strtolower($name) .
                    ($kod ? ", модел $kod" : "") .
                    ($oem ? ", съвместим с OEM: $oem" : "") .
                    ". Високо качество и надеждност.";
        }

        /* KEYWORDS */
        $keywords   = preg_split('/[\s,]+/u', mb_strtolower($name));
        if ($kod)
            $keywords[] = mb_strtolower($kod);
        if ($oem) {
            $parts = preg_split('/[\s,\/]+/', $oem);
            foreach ($parts as $v) {
                $keywords[] = mb_strtolower(trim($v));
            }
        }
        $keywords              = array_unique(array_filter($keywords));
        $data['focus_keyword'] = implode(', ', array_slice($keywords, 0, 12));

        /* SLUG */
        $slugBase = $name;
        if ($oem)
            $slugBase .= " " . $oem;
        if ($kod)
            $slugBase .= " " . $kod;

        $slug             = $this -> transliterateToLatin($slugBase);
        $slug             = strtolower(url_title($slug, '-', true));
        $data['seo_slug'] = $slug ?: "product-$productId";

        /* CANONICAL */
        $data['canonical_url'] = "/" . $data['seo_slug'];

        /* ALWAYS INDEX */
        $data['noindex'] = 0;

        return $data;
    }

    /** @var string Database table name */
    protected $table = '_product_seo';

    /** @var string Primary key for the table */
    protected $primaryKey = 'seo_id';

    /** @var array Fields allowed for mass assignment */
    protected $allowedFields = [
        'product_id', 'seo_title', 'seo_description', 'focus_keyword',
        'canonical_url', 'noindex', 'seo_slug', 'seo_images'
    ];

    /** @var string Return type for queries */
    protected $returnType = 'array';

    /** @var bool Whether timestamps (created_at, updated_at) are automatically managed */
    protected $useTimestamps = false;

    /**
     * Retrieve SEO metadata by product ID.
     *
     * @param int $productId Product identifier
     * @return array|null Returns the SEO record as an associative array, or null if not found
     */
    public function getByProductId(int $productId): ?array {
        return $this -> where('product_id', $productId) -> first();
    }

    /**
     * Insert or update SEO metadata for a product.
     *
     * Automatically generates default SEO values if some fields are missing.
     *
     * @param array $data The SEO data to be saved
     * @return bool True on success, false on failure
     */
    public function saveSEO(array $data): bool {
        if (empty($data['product_id'])) {
            return false;
        }

        // Normalize or fill missing SEO fields with defaults
        $data = $this -> generateDefaultSEO($data);

        $existing = $this -> where('product_id', $data['product_id']) -> first();

        if ($existing) {
            return $this -> update($existing['seo_id'], $data);
        }

        return $this -> insert($data) !== false;
    }

    /**
     * Generate default SEO metadata based on product data.
     *
     * This method ensures all required SEO fields have meaningful defaults:
     * - Title: uses product name
     * - Description: uses short description (trimmed to 160 chars)
     * - Keywords: extracted from product name
     * - Slug: lowercase product name, slugified
     * - Canonical URL: constructed using base URL
     * - noindex: defaults to 0 (indexable)
     *
     * @param array $data Product + SEO data
     * @return array Normalized and completed SEO dataset
     */
    public function generateDefaultSEO(array $data): array {
        helper(['text', 'url']);

        $name      = trim($data['product_name'] ?? '');
        $short     = trim(strip_tags($data['short_description'] ?? ''));
        $long      = trim(strip_tags($data['description'] ?? ''));
        $kod       = trim($data['kod'] ?? '');
        $oem       = trim($data['oem'] ?? '');
        $productId = $data['product_id'] ?? 0;

        /* ============================
          TITLE
          ============================ */
        if (empty($data['seo_title']) && $name) {

            $title = $name;

            if ($kod !== '') {
                $title .= ", КОД: " . $kod;   // ← FIXED
            }

            if ($oem !== '') {
                $title .= ", OEM: " . $oem;
            }

            $data['seo_title'] = $title;
        }

        /* ============================
          DESCRIPTION
          ============================ */
        if (empty($data['seo_description'])) {

            // If description exists → rewrite nicely
            $base = $short ?: $long;

            if ($base !== '') {
                $desc                    = character_limiter($base, 165);
                $data['seo_description'] = rtrim($desc, '.') . '.';
            } else {
                // fallback descriptive text
                $desc = "Предлагаме " . mb_strtolower($name);

                if ($kod !== '') {
                    $desc .= ", модел $kod";
                }
                if ($oem !== '') {
                    $desc .= ", съвместим с OEM: $oem";
                }

                $desc                    .= ". Високо качество и надеждност.";
                $data['seo_description'] = $desc;
            }
        }

        /* ============================
          KEYWORDS
          ============================ */
        if (empty($data['focus_keyword'])) {

            $keywords = [];

            $keywords = array_merge(
                    $keywords,
                    preg_split('/[\s,]+/u', mb_strtolower($name))
            );

            if ($kod !== '') {
                $keywords[] = mb_strtolower($kod);
            }

            if ($oem !== '') {
                foreach (preg_split('/[\s,\/]+/', $oem) as $v) {
                    $keywords[] = mb_strtolower(trim($v));
                }
            }

            $keywords              = array_unique(array_filter($keywords));
            $data['focus_keyword'] = implode(', ', array_slice($keywords, 0, 12));
        }

        /* ============================
          SLUG
          ============================ */
        if (empty($data['seo_slug']) && $name) {

            $slugBase = $name;

            if ($oem !== '') {
                $slugBase .= ' ' . $oem;
            }

            if ($kod !== '') {
                $slugBase .= ' ' . $kod;
            }

            $slug = $this -> transliterateToLatin($slugBase);
            $slug = strtolower(url_title($slug, '-', true));

            $data['seo_slug'] = $slug ?: "product-$productId";
        }

        /* ============================
          CANONICAL (без /product/)
          ============================ */
        if (empty($data['canonical_url'])) {
            $data['canonical_url'] = '/' . ($data['seo_slug'] ?? '');
        }

        /* ============================
          NOINDEX DEFAULT = ALWAYS INDEXED
          ============================ */
        if (!isset($data['noindex']) || $data['noindex'] === null || $data['noindex'] === '') {
            $data['noindex'] = 0;
        }

        return $data;
    }

    /**
     * Transliterate Cyrillic (BG/RU) to Latin characters for SEO slug.
     *
     * @param string $text
     * @return string
     */
    private function transliterateToLatin(string $text): string {
        $map = [
            'а' => 'a', 'б' => 'b', 'в' => 'v', 'г' => 'g', 'д' => 'd', 'е' => 'e', 'ж' => 'zh', 'з' => 'z', 'и' => 'i', 'й' => 'y', 'к' => 'k',
            'л' => 'l', 'м' => 'm', 'н' => 'n', 'о' => 'o', 'п' => 'p', 'р' => 'r', 'с' => 's', 'т' => 't', 'у' => 'u', 'ф' => 'f', 'х' => 'h',
            'ц' => 'ts', 'ч' => 'ch', 'ш' => 'sh', 'щ' => 'sht', 'ъ' => 'a', 'ь' => '', 'ю' => 'yu', 'я' => 'ya',
            'А' => 'A', 'Б' => 'B', 'В' => 'V', 'Г' => 'G', 'Д' => 'D', 'Е' => 'E', 'Ж' => 'Zh', 'З' => 'Z', 'И' => 'I', 'Й' => 'Y', 'К' => 'K',
            'Л' => 'L', 'М' => 'M', 'Н' => 'N', 'О' => 'O', 'П' => 'P', 'Р' => 'R', 'С' => 'S', 'Т' => 'T', 'У' => 'U', 'Ф' => 'F', 'Х' => 'H',
            'Ц' => 'Ts', 'Ч' => 'Ch', 'Ш' => 'Sh', 'Щ' => 'Sht', 'Ъ' => 'A', 'Ь' => '', 'Ю' => 'Yu', 'Я' => 'Ya'
        ];

        return strtr($text, $map);
    }

    public function getSeoList($search = '') {
        $builder = $this -> db -> table($this -> table);
        if ($search) {
            $builder -> like('seo_title', $search)
                    -> orLike('seo_description', $search)
                    -> orLike('focus_keyword', $search);
        }
        return $builder -> get() -> getResultArray();
    }

    public function updateField($id, $field, $value, $extra = []) {
        if (!in_array($field, ['seo_title', 'seo_description', 'focus_keyword', 'canonical_url', 'noindex', 'seo_slug'])) {
            return false;
        }

        $record = $id ? $this -> find($id) : null;

        $fkField = str_contains($this -> table, 'product') ? 'product_id' : 'category_id';
        $fkValue = $extra[$fkField] ?? null;

        if (empty($fkValue)) {
            return false;
        }

        // Ако записът НЕ съществува → създай пълен запис
        if (!$record) {

            $data = [
                $fkField  => $fkValue,
                $field    => $value,
                // ЗАДЪЛЖИТЕЛНО: noindex да е 0 по подразбиране
                'noindex' => ($field === 'noindex' ? $value : 0),
            ];

            return $this -> insert($data, true) ? true : false;
        }

        // Ако записът съществува → update
        return $this -> db -> table($this -> table)
                        -> where($this -> primaryKey, $id)
                        -> set($field, $value)
                        -> update();
    }

}
