<?php

namespace App\Libraries\Gensoft;

use Throwable;

class LIB__GensoftOrder {

    protected static $fb;
    protected $settings;

    public function __construct() {
        helper('utils');

        // Инициализираме service веднъж (глобално за модела)
        if (!self::$fb) {
            self::$fb = service('firebirdConnection');
        }

        // Зареждаме настройките (readonly)
        $this -> settings = (object) (service('settings') -> get('App.gensoft') ?? []);
    }

    /**
     * Entry point – изпращане на поръчка към Gensoft
     */
    public function send(object $order, array $data = []) {
        $settings = $this -> settings; // настройки
        $fb       = self::$fb; // Firebird service
        //
        //try {

        if (!$order) {
            return ['err' => 'Липсва обект поръчка.'];
        }

        $host  = $settings -> gensoftSrv ?? '';
        $host2 = $settings -> gensoftSrv2 ?? '';

        if ($host === '') {
            return ['err' => 'Липсва настройка gensoftSrv.'];
        }

        $ip = h_extractIp($host);
        if (!$fb -> isReachable($ip)) {
            return ['err' => '❌ Няма връзка с Gensoft сървъра.'];
        }

        $skladBase = $_ENV['app.baseSklad'] ?? '';
        $sklad4    = $_ENV['app.sklad4'] ?? '';

        if ($skladBase === '') {
            return ['err' => 'Липсва app.baseSklad.'];
        }

        // ✅ коректна логика за single / multi
        $isSingle      = !empty($data['isSingleProduct']);
        $skladSelected = $isSingle ? (string) ($data['skladName'] ?? $skladBase) : $skladBase;

        $orderArr = $data['order'] ?? [];
        $products = $orderArr['products'] ?? [];

        if (!$products) {
            return ['err' => 'Няма продукти в поръчката.'];
        }

        $konsignacia      = $this -> mapKonsignacia($orderArr['payment_method'] ?? '');
        $orderProductJson = json_decode(($order -> product_json ?? '[]'), true) ?: [];
        $dds              = (float) ($orderArr['dds'] ?? 0);

        $itemsBase   = [];
        $itemsBravas = [];
        $cache       = [];

        foreach ($products as $row) {
            $item = $this -> buildItem(
                    $row,
                    $orderProductJson,
                    $dds,
                    $konsignacia,
                    $skladSelected,
                    $skladBase,
                    $sklad4,
                    $isSingle
            );

            if (isset($item['error'])) {
                return ['err' => $item['error']['message']];
            }

            if (!$item) {
                continue;
            }

            if ($item['isBravas']) {
                $itemsBravas[]             = $item['data'];
                $cache[$row['product_id']] = ['gensoftSmetka' => 'r2'];
            } else {
                $itemsBase[]               = $item['data'];
                $cache[$row['product_id']] = ['gensoftSmetka' => 'r'];
            }
        }

        $resBase = null;
        $resBrav = null;

        if ($itemsBase) {
            $dbh = $fb -> connect($host);
            try {
                $resBase = $this -> insertOrderAndItems($dbh, $itemsBase, $order, $data, true);
            } finally {
                if ($dbh) {
                    ibase_close($dbh);
                }
            }
        }

        if ($itemsBravas) {
            if (!$host2) {
                return ['err' => 'Липсва gensoftSrv2.'];
            }

            $dbh = $fb -> connect($host2);
            try {
                $resBrav = $this -> insertOrderAndItems($dbh, $itemsBravas, $order, $data, false);
            } finally {
                if ($dbh) {
                    ibase_close($dbh);
                }
            }
        }

        if (!$isSingle) {
            return [
                'cacheProdData' => $cache,
                'newOrder'      => 1
            ];
        }

        return $resBrav ?? $resBase ?? ['err' => 'Няма запис в Gensoft.'];
//        } catch (Throwable $e) {
//            return ['err' => 'Gensoft error: ' . $e -> getMessage()];
//        }
    }

    /* =========================================================
     * HELPERS
     * ======================================================= */

    private function mapKonsignacia(string $method): string {
        return match ($method) {
            'B', 'D' => 'D',
            'N', 'E' => 'E',
            default => 'NULL',
        };
    }

    private function buildItem(
            array $row,
            array $orderProductJson,
            float $dds,
            string $konsignacia,
            string $skladSelected,
            string $skladBase,
            string $sklad4,
            bool $isSingle
    ): ?array {
        if (empty($row['product_id']) || empty($row['gensoft_item_id'])) {
            return null;
        }

        $gensoft = json_decode((string) ($row['gensoft_json'] ?? '[]'), true);
        if (!$gensoft || empty($gensoft['grupa'])) {
            throw new \RuntimeException(
                            'Липсва gensoft група за ' . ($row['gensoft_productName'] ?? '')
            );
        }

        $op    = $orderProductJson[$row['product_id']] ?? [];
        $qty   = (float) ($op['qty'] ?? 0);
        $price = (float) ($op['price'] ?? 0);

        if ($qty <= 0) {
            return null;
        }

        if ($dds > 0 && $price > 0) {
            $price += $price * ($dds / 100);
        }

        $isBravas = (!empty($gensoft['isBravas']) && !$isSingle) || ($skladSelected === $sklad4);

        $sklad = $isBravas ? $sklad4 : $skladSelected;

        $kbId = $this -> resolveKbId($konsignacia, $sklad, $sklad4);
        if (!$kbId) {
            return [
                'error' => [
                    'code'    => 'NO_KB_ID',
                    'message' => 'Няма настройка за каса/банка (KB_ID).',
                    'context' => [
                        'konsignacia' => $konsignacia,
                        'sklad'       => $sklad,
                    ]
                ]
            ];
        }

        return [
            'isBravas' => $isBravas,
            'data'     => [
                'stoka'       => $row['gensoft_productName'],
                'grupa'       => $gensoft['grupa'],
                'sklad'       => $sklad,
                'qty'         => $qty,
                'amount'      => $qty * $price,
                'dds'         => $dds,
                'konsignacia' => $konsignacia,
                'kb_id'       => $kbId,
                'ser_no'      => ' ',
                'cena_list'   => 'Цени на едро',
                'enddate'     => date('m-d-Y H:i', strtotime('+7 days')),
            ]
        ];
    }

    private function resolveKbId(string $konsignacia, string $sklad, string $sklad4): ?int {
        $k1 = $this -> settings -> kasaBanki[1] ?? [];
        $k2 = $this -> settings -> kasaBanki[2] ?? [];

        return match ($konsignacia) {
            'D' => (int) ($sklad === $sklad4 ? ($k2['B'] ?? null) : ($k1['B'] ?? null)),
            'E' => (int) ($sklad === $sklad4 ? ($k2['K'] ?? null) : ($k1['K'] ?? null)),
            default => null,
        };
    }

    private function nextSmetka($dbh): string {
        $sql = "SELECT MAX(SMETKA) AS MAX_SM FROM S_ORDER";
        $q   = ibase_query($dbh, $sql);

        if (!$q) {
            return str_pad('1', 10, '0', STR_PAD_LEFT);
        }

        $row = ibase_fetch_object($q);

        $next = (int) ($row -> MAX_SM ?? 0) + 1;

        return str_pad((string) $next, 10, '0', STR_PAD_LEFT);
    }

    private function nextDateTime($dbh): int {
        $sql = "SELECT MAX(DATETIME) AS MAX_DT FROM S_ORDER";
        $q   = ibase_query($dbh, $sql);

        if (!$q) {
            return 1;
        }

        $row = ibase_fetch_object($q);

        return (int) (($row -> MAX_DT ?? 0) + 1);
    }

    private function buildSOrderRow(
            array $data,
            object $order,
            bool $isBase
    ): array {

        $orderArr = $data['order'] ?? [];

        $delivery_json      = json_decode((string) ($order -> delivery_json ?? '[]'), true) ?: [];
        $tovaritelniza_json = json_decode((string) ($order -> tovaritelniza_json ?? '[]'), true) ?: [];

        $deliveryMethod = (string) ($order -> delivery_method ?? '');
        $whoPayBill     = (string) ($order -> whoPayBill ?? '');
        $prevozvach     = (string) ($order -> prevozvach ?? '');
        $tovarNo        = (string) ($order -> tovaritelNo ?? '');

        /* ==============================
         * NOTE (Бележка)
         * ============================== */
        $note = ' ';

        if (!empty($order -> belezka_private)) {
            $note .= ' Бележка: ' . $order -> belezka_private;
        }

        if (!empty($order -> invoice_tip)) {
            $note .= ' Документ: ' . $order -> invoice_tip;
        }

        if (!empty($order -> belezka)) {
            $note .= ' Коментар: ' . $order -> belezka;
        }

        if ($deliveryMethod === 'curier') {
            $note .= ' За чия сметка: ' . $whoPayBill;
        }

        if (!empty($delivery_json)) {

            $kurier = $delivery_json['izborKurier'] ?? '';

            foreach ($delivery_json as $k => $v) {
                $v = (string) $v;

                $note .= match ($k) {
                    'lice_zaKont' => ', лице за контакт: ' . $v,
                    'tel' => ', тел.: ' . $v,
                    'grad' => ', ' . $v,
                    'ofis' => ', офис: ' . $v,
                    default => ''
                };
            }

            if (str_contains($kurier, 'econt')) {
                $note .= ', доставка с Еконт';
            } elseif (str_contains($kurier, 'speedy')) {
                $note .= ', доставка със Спиди';
            }
        }

        if (!empty($tovaritelniza_json['payment'])) {
            $note .= ' Плащане: ' . match ($tovaritelniza_json['payment']) {
                        'za_smetkaPoluchatel' => 'За сметка на получателя',
                        'za_smetkaMe' => 'За сметка на продавача',
                        'za_smetkaShare' => 'Споделени разходи',
                        default => ''
                    };
        }

        if ($tovarNo !== '') {
            $note .= ' ТВ: ' . $tovarNo;
        }

        /* ==============================
         * PREVOZ
         * ============================== */
        $prevoz = ' ';

        if ($deliveryMethod === 'curier') {
            $kurier = $delivery_json['izborKurier'] ?? '';

            if (str_contains($kurier, 'econt')) {
                $prevoz = $this -> settings -> prevoz['E'] ?? 'ЕКОНТ';
            } elseif (str_contains($kurier, 'speedy')) {
                $prevoz = $this -> settings -> prevoz['S'] ?? 'СПИДИ';
            }
        }

        if ($deliveryMethod === 'firmCar') {
            $prevoz = $this -> settings -> prevoz['F'] ?? 'ФИРМЕН ТРАНСПОРТ';
        }

        if ($deliveryMethod === 'selfCar') {
            $prevoz = $this -> settings -> prevoz['ST'] ?? 'ЛИЧЕН ТРАНСПОРТ';
        }

        if ($prevozvach !== '') {
            $prevoz = $prevozvach;
        }

        /* ==============================
         * ОСНОВНИ ПОЛЕТА
         * ============================== */

        $dtDoc = h_dateToGenDT(new \DateTime('now', new \DateTimeZone('UTC')));

        $firma = h_CP1251((string) ($orderArr['klient'] ?? ' '));
        $agent = h_CP1251((string) ($order -> agent ?? ' '));
        $owner = h_CP1251((string) ($orderArr['owner'] ?? ' '));

        /* ==============================
         * PROPS (base / bravas)
         * ============================== */
        $propsKey = $isBase ? ($this -> settings -> props['key'] ?? '') : ($this -> settings -> props['key2'] ?? '');

        $propsVal = $isBase ? ($this -> settings -> props['val'] ?? '') : ($this -> settings -> props['val2'] ?? '');

        $props = h_CP1251("|{$propsKey}—{$propsVal}|");

        /* ==============================
         * SHIP DATE
         * ============================== */
        $shipDate = '';

        if (!empty($order -> date_shipping)) {
            $shipDate = date('Y-m-d', strtotime($order -> date_shipping)) . ' ' . date('H:i');
        }

        return [
            'DT_DOC'    => $dtDoc,
            'FIRMA'     => trim($firma),
            'PREWOZ'    => trim(substr(h_CP1251($prevoz), 0, 48)),
            'AGENT'     => trim(substr($agent, 0, 48)),
            'OWNER'     => trim(substr($owner, 0, 48)),
            'PROPS'     => $props,
            'KASA'      => 'I',
            'SHIP_DATE' => $shipDate,
            'NOTE'      => trim(substr(h_CP1251N($note), 0, 254)) ?: ' ',
        ];
    }

    private function countOrderItems($dbh, int $datetime): int {
        if (!$dbh || $datetime <= 0) {
            return 0;
        }

        $sql = "SELECT COUNT(*) AS CNT
            FROM S_ORDER_ITEMS
            WHERE DATETIME = {$datetime}";

        $q = ibase_query($dbh, $sql);

        if (!$q) {
            return 0;
        }

        $row = ibase_fetch_object($q);

        return (int) ($row -> CNT ?? 0);
    }

    private function insertOrderAndItems(
            $dbh,
            array $items,
            object $order,
            array $data,
            bool $isBase): array {

        if (!$items) {
            return [
                'err' => [
                    'code'    => 'NO_ITEMS',
                    'message' => 'Няма артикули за запис.'
                ]
            ];
        }

        $gensoftOrder = json_decode(($order -> gensoftOrder_json ?? '[]'), true) ?: [];

        $datetimeKey = $isBase ? 'datetime' : 'datetime2';
        $smetkaKey   = $isBase ? 'smetka' : 'smetka2';

        $datetime = $gensoftOrder[$datetimeKey] ?? null;
        $smetka   = $gensoftOrder[$smetkaKey] ?? null;

        // нови стойности ако няма запис
        if (!$datetime || !$smetka) {
            $datetime = $this -> nextDateTime($dbh);
            $smetka   = $this -> nextSmetka($dbh);
        }

        $tr = ibase_trans($dbh, IBASE_DEFAULT);
        if (!$tr) {
            return [
                'err' => [
                    'code'    => 'NO_TRANSACTION',
                    'message' => 'Не може да се отвори Firebird транзакция.'
                ]
            ];
        }


        // 1️⃣ INSERT S_ORDER (само ако е нова)
        if (empty($gensoftOrder[$datetimeKey])) {

            $orderRow = $this -> buildSOrderRow($data, $order, $isBase);

            $stmtOrder = ibase_prepare(
                    $tr,
                    'INSERT INTO S_ORDER
                (DATETIME, DT_DOC, SMETKA, FIRMA, PREWOZ, AGENT, OWNER, PROPS, KASA, SHIP_DATE, NOTE)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)'
            );

            ibase_execute(
                    $stmtOrder,
                    (int) $datetime,
                    $orderRow['DT_DOC'],
                    $smetka,
                    $orderRow['FIRMA'],
                    $orderRow['PREWOZ'],
                    $orderRow['AGENT'],
                    $orderRow['OWNER'],
                    $orderRow['PROPS'],
                    $orderRow['KASA'],
                    $orderRow['SHIP_DATE'],
                    $orderRow['NOTE']
            );
        }

        // 2️⃣ стартов NUM (ако вече има items)
        $num = 1;
        if (!empty($gensoftOrder[$datetimeKey])) {
            $num = $this -> countOrderItems($dbh, (int) $datetime) + 1;
        }

        // 3️⃣ INSERT ITEMS
        $stmtItem = ibase_prepare($tr,
                'INSERT INTO S_ORDER_ITEMS(
                    DATETIME, 
                    STOKA, 
                    GRUPA, 
                    SKLAD,
                    KOLICH, 
                    CENA, 
                    NUM, 
                    DDS, 
                    KONSIGNACIA, 
                    VALUTA,
                    KB_ID, 
                    SER_NO, 
                    CENA_LIST, 
                    ENDDATE) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)'
        );
        dd($items);
        foreach ($items as $item) {
 
            ibase_execute(
                    $stmtItem,
                    (int) $datetime,
                    h_CP1251N($item['stoka']),
                    h_CP1251N($item['grupa']),
                    h_CP1251N($item['sklad']),
                    (float) $item['qty'],
                    (float) $item['amount'],
                    (int) $num,
                    (float) $item['dds'],
                    h_CP1251N($item['konsignacia']),
                     "'M'",
                    (int) $item['kb_id'],
                    h_CP1251N($item['ser_no'] ?? ' '),
                    h_CP1251N($item['cena_list'] ?? ' '),
                    "'" . $item['enddate'] . "'"
            );

            $num++;
        }

        ibase_commit($tr);

        return [
            'newOrder' => 1,
            'smetka'   => $smetka
        ];
    }

}
