<?php
namespace App\Controllers\RefillStock;

use App\Controllers\BaseController;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;

class RefillStock extends BaseController
{
    public function index()
    {
        return view('RefillStock/index', ['title' => 'Дозареждане на стока']);
    }

    public function data()
    {
        include APPPATH . 'Controllers/RefillStock/RefillStockData.php';

        $group     = $this->request->getGet('group');
        $product   = $this->request->getGet('product');
        $warehouse = $this->request->getGet('warehouse');
        $months    = (int)($this->request->getGet('months') ?? 6);

        $allGroups = [];
        $allProducts = [];
        $allWarehouses = [];
        foreach ($refillStockData as $r) {
            $allGroups[$r['Група']] = true;
            $allProducts[$r['Стока']] = true;
            $allWarehouses[$r['Склад']] = true;
        }

        $filtered = array_filter($refillStockData, function ($r) use ($group, $product, $warehouse) {
            return (!$group     || $r['Група'] === $group)
                && (!$product   || $r['Стока'] === $product)
                && (!$warehouse || $r['Склад'] === $warehouse);
        });

        $activeWarehouses = [];
        foreach ($filtered as $r) {
            $activeWarehouses[$r['Склад']] = true;
        }
        $activeWarehouses = array_keys($activeWarehouses);
        sort($activeWarehouses);

        $byProduct = [];
        foreach ($filtered as $r) {
            $key = $r['Стока'];
            if (!isset($byProduct[$key])) {
                $byProduct[$key] = [
                    'Група' => $r['Група'],
                    'Стока' => $r['Стока'],
                    'Цена без ДДС' => (float)$r['Цена без ДДС'],
                    'Продадени' => 0,
                    'Налични' => 0,
                    'warehouses' => []
                ];
            }
            $byProduct[$key]['Продадени'] += (int)$r['Продадени'];
            $byProduct[$key]['Налични']   += (int)$r['Налични'];

            if (!isset($byProduct[$key]['warehouses'][$r['Склад']])) {
                $byProduct[$key]['warehouses'][$r['Склад']] = ['sold' => 0, 'stock' => 0];
            }
            $byProduct[$key]['warehouses'][$r['Склад']]['sold']  += (int)$r['Продадени'];
            $byProduct[$key]['warehouses'][$r['Склад']]['stock'] += (int)$r['Налични'];
        }

        foreach ($byProduct as &$row) {
            $avg = $row['Продадени'] / max($months, 1);
            $row['Количество за доставка'] = 0;
        }
        unset($row);

        $items = array_values($byProduct);
        usort($items, fn($a, $b) => strcasecmp($a['Стока'], $b['Стока']));

        return $this->response->setJSON([
            'items' => $items,
            'warehouses' => $activeWarehouses,
            'options' => [
                'groups' => array_keys($allGroups),
                'products' => array_keys($allProducts),
                'warehouses' => array_keys($allWarehouses),
            ],
        ]);
    }

    public function export()
    {
        $json = $this->request->getJSON(true);
    
        if (!$json || !is_array($json)) {
            return $this->response->setStatusCode(400)->setBody('Няма данни за експорт.');
        }
    
        $rows = [];
        foreach ($json as $row) {
            $qty = isset($row['qty']) ? (int)$row['qty'] : 0;
            if ($qty > 0) {
                $rows[] = [
                    $row['group'],
                    $row['product'],
                    (float)$row['price'],
                    (int)$row['sold'],
                    (int)$row['stock'],
                    $qty
                ];
            }
        }
    
        if (empty($rows)) {
            return $this->response->setStatusCode(204)->setBody('Няма редове за експорт.');
        }
    
        if (!class_exists(Spreadsheet::class)) {
            return $this->response->setStatusCode(500)
                ->setBody('Липсва PhpSpreadsheet. Инсталирай: composer require phpoffice/phpspreadsheet');
        }
    
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        $sheet->setTitle('Дозареждане');
    
        $sheet->mergeCells('A1:F1');
        $sheet->setCellValue('A1', 'VALPERS – Отчет за складови наличности');
        $sheet->getStyle('A1')->applyFromArray([
            'font' => ['bold' => true, 'size' => 18, 'color' => ['rgb' => '1E3A8A']],
            'alignment' => ['horizontal' => \PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_CENTER],
        ]);
    
        $sheet->mergeCells('A2:F2');
        $sheet->setCellValue('A2', 'Дозареждане на стока – лист на продукти');
        $sheet->getStyle('A2')->applyFromArray([
            'font' => ['size' => 12, 'color' => ['rgb' => '374151']],
            'alignment' => ['horizontal' => \PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_CENTER],
        ]);
    
        $sheet->mergeCells('A3:F3');
    
        $headers = ['Група', 'Стока', 'Цена от последна доставка без ДДС', 'Продадени', 'Налични', 'Количество за доставка'];
        $sheet->fromArray([$headers], null, 'A4');
        $sheet->fromArray($rows, null, 'A5');
    
        $sheet->getStyle("A4:F4")->applyFromArray([
            'font' => ['bold' => true, 'color' => ['rgb' => 'FFFFFF']],
            'fill' => ['fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID, 'color' => ['rgb' => '2563EB']],
            'alignment' => ['horizontal' => \PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_CENTER],
            'borders' => ['allBorders' => ['borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN]]
        ]);
    
        $lastRow = count($rows) + 4;
        $sheet->getStyle("A4:F{$lastRow}")->applyFromArray([
            'borders' => ['allBorders' => ['borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_HAIR]]
        ]);
    
        for ($i = 5; $i <= $lastRow; $i++) {
            $sheet->getStyle("A{$i}:F{$i}")->applyFromArray([
                'fill' => [
                    'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                    'color' => ['rgb' => 'D4FCD4']
                ],
            ]);
        }
    
        $sheet->getStyle("C5:C{$lastRow}")->getNumberFormat()->setFormatCode('#,##0.00" лв."');
        $sheet->getStyle("D5:F{$lastRow}")->getNumberFormat()->setFormatCode('0');
    
        foreach (range('A', 'F') as $col) {
            $sheet->getColumnDimension($col)->setAutoSize(true);
        }
    
        $sheet->setCellValue("A" . ($lastRow + 2), 'Дата на експорт: ' . date('d.m.Y H:i'));
    
        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment; filename="dozarejdane_na_stoka.xlsx"');
    
        (new Xlsx($spreadsheet))->save('php://output');
        exit;
    }
}
