<?php

namespace App\Controllers\Dashboard;

use App\Controllers\BaseController;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;

class Dashboard extends BaseController
{
   public function index()
    {
        $salesData = include __DIR__ . '/DashboardData.php';
    
        $now = new \DateTime();
        $start = (clone $now)->modify('-3 years');
    
        $filtered = array_filter($salesData, function ($row) use ($start, $now) {
            $d = new \DateTime($row['date']);
            return $d >= $start && $d <= $now;
        });
    
        $agentTotals = [];
        foreach ($filtered as $row) {
            $agentTotals[$row['agent']] = ($agentTotals[$row['agent']] ?? 0) + $row['profit'];
        }
        arsort($agentTotals);
    
        $totalProfit = array_sum($agentTotals);
    
        $summary = [];
        foreach ($agentTotals as $agentName => $profit) {
            $summary[] = [
                'agent' => $agentName,
                'total' => round($profit, 2),
                'share' => $totalProfit > 0 ? round(($profit / $totalProfit) * 100, 2) : 0,
            ];
        }
    
        // 💰 Общо финансови стойности
        $totalExpense = array_sum(array_column($filtered, 'expense'));
        $totalIncome  = array_sum(array_column($filtered, 'income'));
        $totalProfit  = array_sum(array_column($filtered, 'profit'));
        $avgMargin    = round(array_sum(array_column($filtered, 'margin')) / max(count($filtered), 1), 2);
    
        // 👤 Агенти за филтър
        $agents = array_unique(array_map(fn($row) => $row['agent'], $filtered));
        sort($agents);
    
        // 🔹 Генерираме и графични данни (ако ги ползваш)
        $chartData = $this->prepareChartData($salesData, 'year');
    
        return view('Dashboard/dashboard', [
            'title' => 'Общо приходи (3 години назад)',
            'finance' => [
                'expense' => $totalExpense,
                'income'  => $totalIncome,
                'profit'  => $totalProfit,
                'margin'  => $avgMargin,
            ],
            'chartData' => $chartData,
            'agents' => $agents,
            'summary' => $summary,
            'summary_total' => $totalProfit,
            'startDate' => $start->format('Y-m-d'),
            'endDate'   => $now->format('Y-m-d'),
        ]);
    }
    
        public function data($period = 'day')
        {
            $salesData = include __DIR__ . '/DashboardData.php';
            $chartData = $this->prepareChartData($salesData, $period);
            return $this->response->setJSON($chartData);
        }
    
        private function prepareChartData(array $salesData, string $period = 'day'): array
        {
            $now = new \DateTime();
    
            switch ($period) {
                case 'week':
                    $start = (clone $now)->modify('-6 days');
                    $interval = 'P1D';
                    $format = 'd.m';
                    break;
                case 'month':
                    $start = (clone $now)->modify('-29 days');
                    $interval = 'P1D';
                    $format = 'd.m';
                    break;
                case 'year':
                    $start = (clone $now)->modify('-11 months');
                    $interval = 'P1M';
                    $format = 'M Y';
                    break;
                default:
                    $start = (clone $now)->setTime(0, 0);
                    $interval = 'P1D';
                    $format = 'd.m';
                    break;
            }
    
            $filtered = array_filter($salesData, function ($row) use ($start, $now) {
                $d = new \DateTime($row['date']);
                return $d >= $start && $d <= $now;
            });
    
            $grouped = [];
            foreach ($filtered as $row) {
                $key = (new \DateTime($row['date']))->format($format);
                if (!isset($grouped[$key])) {
                    $grouped[$key] = ['income' => 0, 'expense' => 0, 'profit' => 0];
                }
                $grouped[$key]['income']  += $row['income'];
                $grouped[$key]['expense'] += $row['expense'];
                $grouped[$key]['profit']  += $row['profit'];
            }
    
            $range = new \DatePeriod($start, new \DateInterval($interval), (clone $now)->modify('+1 day'));
            foreach ($range as $d) {
                $key = $d->format($format);
                $grouped[$key] ??= ['income' => 0, 'expense' => 0, 'profit' => 0];
            }
    
            ksort($grouped);
    
            $labels  = array_keys($grouped);
            $income  = array_map(fn($r) => round($r['income'], 2), $grouped);
            $expense = array_map(fn($r) => round($r['expense'], 2), $grouped);
            $profit  = array_map(fn($r) => round($r['profit'], 2), $grouped);
            $total   = round(array_sum($profit), 2);
    
            return [
                'labels'  => $labels,
                'income'  => array_values($income),
                'expense' => array_values($expense),
                'profit'  => array_values($profit),
                'total'   => $total,
            ];
        }
    
    public function dataCustom()
    {
        $salesData = include __DIR__ . '/DashboardData.php';
    
        $from  = $this->request->getGet('from');
        $to    = $this->request->getGet('to');
        $agent = $this->request->getGet('agent');
    
        if (!$from || !$to) {
            return $this->response->setJSON(['error' => 'Моля, изберете начална и крайна дата!']);
        }
    
        $start = new \DateTime($from);
        $end   = new \DateTime($to);
    
        $filtered = array_filter($salesData, function ($item) use ($start, $end, $agent) {
            $d = new \DateTime($item['date']);
            if ($agent && $item['agent'] !== $agent) return false;
            return $d >= $start && $d <= $end;
        });
    

        $grouped = [];
        foreach ($filtered as $row) {
            $key = (new \DateTime($row['date']))->format('d.m.Y');
            $grouped[$key]['income'] = ($grouped[$key]['income'] ?? 0) + (float)$row['income'];
            $grouped[$key]['profit'] = ($grouped[$key]['profit'] ?? 0) + (float)$row['profit'];
        }
        ksort($grouped);
    
        $result = [
            'labels' => array_keys($grouped),
            'income' => array_column($grouped, 'income'),
            'profit' => array_column($grouped, 'profit'),
            'total'  => round(array_sum(array_column($grouped, 'profit')), 2),
        ];
    

        if (!$agent) {
            require_once __DIR__ . '/FirmsData.php';
    
            // 1️⃣ Суми по агент
            $agentData = [];
            foreach ($filtered as $row) {
                $a = $row['agent'];
                if (!isset($agentData[$a])) {
                    $agentData[$a] = ['profit' => 0, 'income' => 0];
                }
                $agentData[$a]['profit'] += (float)$row['profit'];
                $agentData[$a]['income'] += (float)$row['income'];
            }
    

            $summary = [];
            foreach ($agentData as $agentName => $vals) {
                $clientType = null;
    
                foreach ($inactiveFirms as $firm) {
                    if ($firm['agent'] === $agentName && !empty($firm['client_type'])) {
                        $clientType = $firm['client_type'];
                        break;
                    }
                }
                if (!$clientType && isset($activeFirms)) {
                    foreach ($activeFirms as $firm) {
                        if ($firm['agent'] === $agentName && !empty($firm['client_type'])) {
                            $clientType = $firm['client_type'];
                            break;
                        }
                    }
                }
    
                $summary[] = [
                    'agent'       => $agentName,
                    'client_type' => $clientType ?: 'Неопределен',
                    'profit'      => round($vals['profit'], 2),
                    'turnover'    => round($vals['income'], 2),
                ];
            }
    
            // 3️⃣ Изчисляваме общо печалба и общ оборот
            $result['summary'] = $summary;
            $result['summary_total'] = array_sum(array_column($summary, 'profit'));
            $result['summary_total_turnover'] = array_sum(array_column($summary, 'turnover'));
        }
    
        return $this->response->setJSON($result);
    }
        
    public function firms()
    {
        require __DIR__ . '/FirmsData.php';
    
        if (!isset($activeFirms) || !isset($inactiveFirms)) {
            return $this->response->setJSON(['error' => 'Не са намерени променливите $activeFirms и $inactiveFirms']);
        }
    
        $agents = array_unique(array_merge(
            array_column($activeFirms, 'agent'),
            array_column($inactiveFirms, 'agent')
        ));
        sort($agents);
    
        $groups = array_unique(array_merge(
            array_column($activeFirms, 'group_main'),
            array_column($inactiveFirms, 'group_main')
        ));
        sort($groups);
    
        // 🔹 типове клиенти — събираме всички от активни и неактивни
        $clientTypes = array_unique(array_merge(
            array_column($activeFirms, 'client_type'),
            array_column($inactiveFirms, 'client_type')
        ));
        sort($clientTypes);
    
        return $this->response->setJSON([
            'activeFirms'   => $activeFirms,
            'inactiveFirms' => $inactiveFirms,
            'agents'        => $agents,
            'groups'        => $groups,
            'clientTypes'   => $clientTypes
        ]);
    }
    public function inactiveFiltered()
    {
        require __DIR__ . '/FirmsData.php';
    
        $from   = $this->request->getGet('from');
        $to     = $this->request->getGet('to');
        $agent  = $this->request->getGet('agent');
        $group  = $this->request->getGet('group');
        $type   = $this->request->getGet('client_type'); // 🆕 филтър по тип клиент
    
        $filtered = array_filter($inactiveFirms, function ($firm) use ($from, $to, $agent, $group, $type) {
            $date = new \DateTime($firm['last_income_date']);
            if ($from && $date < new \DateTime($from)) return false;
            if ($to && $date > new \DateTime($to)) return false;
            if ($agent && $firm['agent'] !== $agent) return false;
            if ($group && stripos($firm['group_main'], $group) === false) return false;
            if ($type && (!isset($firm['client_type']) || $firm['client_type'] !== $type)) return false; // 🆕
            return true;
        });
    
        // 🔹 Изкарваме резултата като JSON (вече съдържа client_type)
        return $this->response->setJSON(array_values($filtered));
    }
    
    
    public function overviewGroups()
    {
        $groups = include __DIR__ . '/GroupsData.php';
    
        $labels   = [];
        $profits  = [];
        $turnover = [];
    
        foreach ($groups as $g) {
            $labels[]   = $g['group'];
            $profits[]  = round($g['profit'], 2);
            $turnover[] = round($g['turnover'], 2);
        }
    
        return $this->response->setJSON([
            'labels'   => $labels,
            'profit'   => $profits,
            'turnover' => $turnover,
        ]);
    }
    
  public function stockMovement()
{
    include APPPATH . 'Controllers/Dashboard/StockMovementData.php';

    $from     = $this->request->getGet('from');
    $to       = $this->request->getGet('to');
    $group    = $this->request->getGet('group');
    $subGroup = $this->request->getGet('sub');  // <-- НОВО !!!

    $data = $stockData;

    // Филтър: период
    $fromDate = $from ? \DateTime::createFromFormat('Y-m-d', $from) : null;
    $toDate   = $to ? \DateTime::createFromFormat('Y-m-d', $to) : null;

    if ($fromDate && $toDate) {
        $data = array_filter($data, function ($r) use ($fromDate, $toDate) {
            $d = \DateTime::createFromFormat('m/d/Y', $r['От дата']);
            return $d && $d >= $fromDate && $d <= $toDate;
        });
    }

    // Филтър: главна група
    if ($group) {
        $data = array_filter($data, fn($r) => $r['Главна група'] === $group);
    }

    // Филтър: подгрупа  <-- НОВО !!!
    if ($subGroup) {
        $data = array_filter($data, fn($r) => $r['Подгрупа'] === $subGroup);
    }

    // Групиране по продукт
    $products = [];
    foreach ($data as $r) {

        $key = $r['Стока'];

        $qty        = (float)$r['Количество'];
        $turnover   = (float)$r['Оборот'];
        $profit     = (float)$r['Чист приход'];
        $cost       = $turnover - $profit;            // Доставна цена за 1 бр
        $totalCost  = $qty * $cost;                   // ОБЩ ТОТАЛ

        if (!isset($products[$key])) {
            $products[$key] = [
                'Главна група' => $r['Главна група'],
                'Подгрупа'     => $r['Подгрупа'],
                'Стока'        => $r['Стока'],
                'Наличност'    => 0,
                'Доставна цена'=> $cost,
                'ОБЩ ТОТАЛ'    => 0,
                'Продадени'    => 0,
                'Оборот'       => 0,
                'Чист приход'  => 0,
                'Последна доставка' => $r['От дата'],
            ];
        }

        $products[$key]['Наличност']    += $qty;
        $products[$key]['Продадени']    += $qty;
        $products[$key]['Оборот']       += $turnover;
        $products[$key]['Чист приход']  += $profit;
        $products[$key]['ОБЩ ТОТАЛ']    += $totalCost;

        $d = \DateTime::createFromFormat('m/d/Y', $r['От дата']);
        $last = \DateTime::createFromFormat('m/d/Y', $products[$key]['Последна доставка']);

        if ($d > $last) {
            $products[$key]['Последна доставка'] = $r['От дата'];
        }
    }

    return $this->response->setJSON([
        'products' => array_values($products)
    ]);
}


    
    public function bestSellers()
    {
        include APPPATH . 'Controllers/Dashboard/StockMovementData.php';
    
        $from  = $this->request->getGet('from');
        $to    = $this->request->getGet('to');
        $group = $this->request->getGet('group');
    
        $data = $stockData;
    
        $fromDate = $from ? \DateTime::createFromFormat('Y-m-d', $from) : null;
        $toDate   = $to ? \DateTime::createFromFormat('Y-m-d', $to) : null;
    
        if ($fromDate && $toDate) {
            $data = array_filter($data, function ($r) use ($fromDate, $toDate) {
                $d = \DateTime::createFromFormat('m/d/Y', $r['От дата']);
                return $d && $d >= $fromDate && $d <= $toDate;
            });
        }
    
        if ($group) {
            $data = array_filter($data, fn($r) => $r['Главна група'] === $group);
        }
    
        $products = [];
        foreach ($data as $r) {
            $product = $r['Стока'];
            if (!isset($products[$product])) {
                $products[$product] = [
                    'Продукт' => $product,
                    'Група' => $r['Главна група'],
                    'Количество' => 0,
                    'Оборот' => 0,
                    'Чист приход' => 0,
                ];
            }
            $products[$product]['Количество'] += (float)$r['Количество'];
            $products[$product]['Оборот'] += (float)$r['Оборот'];
            $products[$product]['Чист приход'] += (float)$r['Чист приход'];
        }
    
        usort($products, fn($a, $b) => $b['Чист приход'] <=> $a['Чист приход']);
    
        $top = array_slice($products, 0, 10);
    
        return $this->response->setJSON(['products' => $top]);
    }
    
    public function debts()
    {
        $file = __DIR__ . '/DebtorsSummary.php';
        $debts = file_exists($file) ? include $file : [];
        return $this->response->setJSON($debts);
    }
    
    public function loadUsers()
    {
        $db = \Config\Database::connect();
        $query = $db->query("SELECT id, username, first_name, last_name FROM users WHERE active = 1");
        return $this->response->setJSON($query->getResultArray());
    }
    
    public function loadFirmsFromDB()
    {
        $db = \Config\Database::connect();
        $query = $db->query("SELECT klient_id, klient_name, klient_grad, klient_adres, klient_tel FROM _klient WHERE isActive='Y' LIMIT 30");
        return $this->response->setJSON($query->getResultArray());
    }
    
    public function exportStockMovement()
    {
        include APPPATH . 'Controllers/Dashboard/StockMovementData.php';
    
        $from  = $this->request->getGet('from');
        $to    = $this->request->getGet('to');
        $group = $this->request->getGet('group');
        $sub   = $this->request->getGet('sub');
    
        $rows = $this->calculateStockMovement($stockData, $from, $to, $group, $sub);
    
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        $sheet->setTitle("Движение на стока");
    
        // HEADER NAMES
        $headers = [
            "A1" => "Главна група",
            "B1" => "Подгрупа",
            "C1" => "Стока",
            "D1" => "Наличност",
            "E1" => "Доставна цена",
            "F1" => "ОБЩ ТОТАЛ БЕЗ ДДС",
            "G1" => "Продадени",
            "H1" => "Оборот",
            "I1" => "Чист приход",
            "J1" => "Последна доставка",
        ];
    
        // Apply header values + styling
        foreach ($headers as $cell => $text) {
            $sheet->setCellValue($cell, $text);
        }
    
        // BEAUTIFUL HEADER STYLE
        $sheet->getStyle("A1:J1")->getFont()->setBold(true)->setSize(12);
        $sheet->getStyle("A1:J1")->getAlignment()->setHorizontal("center");
        $sheet->getStyle("A1:J1")->getFill()
            ->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)
            ->getStartColor()->setARGB("FF4C6EF5");
        $sheet->getStyle("A1:J1")->getFont()->getColor()->setARGB("FFFFFFFF");
    
        $row = 2;
        foreach ($rows as $r) {
    
            $sheet->setCellValue("A{$row}", $r['Главна група']);
            $sheet->setCellValue("B{$row}", $r['Подгрупа']);
            $sheet->setCellValue("C{$row}", $r['Стока']);
            $sheet->setCellValue("D{$row}", $r['Наличност']);
            $sheet->setCellValue("E{$row}", $r['Доставна цена']);
            $sheet->setCellValue("F{$row}", $r['ОБЩ_ТОТАЛ']);
            $sheet->setCellValue("G{$row}", $r['Продадени']);
            $sheet->setCellValue("H{$row}", $r['Оборот']);
            $sheet->setCellValue("I{$row}", $r['Чист приход']);
            $sheet->setCellValue("J{$row}", $r['Последна доставка']);
    
            // ROW STRIPING (even rows light gray)
            if ($row % 2 == 0) {
                $sheet->getStyle("A{$row}:J{$row}")->getFill()
                    ->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)
                    ->getStartColor()->setARGB("FFF3F4F8");
            }
    
            $row++;
        }
    
        // Borders for all cells
        $sheet->getStyle("A1:J" . ($row-1))->getBorders()->getAllBorders()
            ->setBorderStyle(\PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN);
    
        // Auto-size
        foreach (range('A','J') as $col) {
            $sheet->getColumnDimension($col)->setAutoSize(true);
        }
    
        // AutoFilter
        $sheet->setAutoFilter("A1:J1");
    
        // OUTPUT
        header("Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
        header("Content-Disposition: attachment; filename=\"movement_beautiful.xlsx\"");
        header("Cache-Control: max-age=0");
    
        $writer = new Xlsx($spreadsheet);
        $writer->save("php://output");
    }
    
    
    
    public function exportBestSellers()
    {
        include APPPATH . 'Controllers/Dashboard/StockMovementData.php';
    
        $from  = $this->request->getGet('from');
        $to    = $this->request->getGet('to');
        $group = $this->request->getGet('group');
        $sub   = $this->request->getGet('sub');
    
        $rows = $this->calculateBestSellers($stockData, $from, $to, $group, $sub);
    
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        $sheet->setTitle("Най-продавани");
    
        // Header text
        $headers = [
            "A1" => "Продукт",
            "B1" => "Група",
            "C1" => "Продадени",
            "D1" => "Оборот",
            "E1" => "Чист приход",
        ];
    
        foreach ($headers as $cell => $name) {
            $sheet->setCellValue($cell, $name);
        }
    
        // Beautiful header styling
        $sheet->getStyle("A1:E1")->getFont()->setBold(true)->setSize(12);
        $sheet->getStyle("A1:E1")->getAlignment()->setHorizontal("center");
        $sheet->getStyle("A1:E1")->getFill()
            ->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)
            ->getStartColor()->setARGB("FFFFA500");
    
        $sheet->getStyle("A1:E1")->getFont()->getColor()->setARGB("FF000000");
    
        $row = 2;
        foreach ($rows as $p) {
    
            $sheet->setCellValue("A{$row}", $p['Продукт']);
            $sheet->setCellValue("B{$row}", $p['Група']);
            $sheet->setCellValue("C{$row}", $p['Продадени']);
            $sheet->setCellValue("D{$row}", $p['Оборот']);
            $sheet->setCellValue("E{$row}", $p['Чист приход']);
    
            // Gray striping every second row
            if ($row % 2 == 0) {
                $sheet->getStyle("A{$row}:E{$row}")->getFill()
                    ->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)
                    ->getStartColor()->setARGB("FFF5F5F5");
            }
    
            $row++;
        }
    
        // Borders for table
        $sheet->getStyle("A1:E".($row-1))->getBorders()->getAllBorders()
            ->setBorderStyle(\PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN);
    
        // Auto-size columns
        foreach (range('A','E') as $col) {
            $sheet->getColumnDimension($col)->setAutoSize(true);
        }
    
        // Enable filtering
        $sheet->setAutoFilter("A1:E1");
    
        header("Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
        header("Content-Disposition: attachment; filename=\"best_beautiful.xlsx\"");
        header("Cache-Control: max-age=0");
    
        $writer = new Xlsx($spreadsheet);
        $writer->save("php://output");
    }
    
    
    
    public function exportDeliveries()
    {
        $deliveriesData = include APPPATH . 'Controllers/Dashboard/DeliveriesData.php';
    
        $monthsBack     = (int)($this->request->getGet('months') ?? 6);
        $deliveryPeriod = (int)($this->request->getGet('deliveryPeriod') ?? $monthsBack);
        $percent        = (int)($this->request->getGet('percent') ?? 10);
        $group          = $this->request->getGet('group');
    
        if (!$deliveriesData) return;
    
        // === 1) Намираме най-новата дата ===
        $latestDate = null;
        foreach ($deliveriesData as $row) {
            $d = \DateTime::createFromFormat('m/d/Y', $row['От дата']);
            if ($d && (!$latestDate || $d > $latestDate)) $latestDate = $d;
        }
        if (!$latestDate) return;
    
        $fromDate    = (clone $latestDate)->modify("-{$monthsBack} months");
        $staleBefore = (clone $latestDate)->modify("-{$deliveryPeriod} months");
    
        // === 2) Агрегиране ===
        $aggregated = [];
    
        foreach ($deliveriesData as $r) {
            $d = \DateTime::createFromFormat('m/d/Y', $r['От дата']);
            if (!$d) continue;
    
            if ($group && $r['Главна група'] !== $group && $r['Група'] !== $group) continue;
            if ($d < $fromDate) continue;
    
            if ($percent > 0 && $r['Налични'] > ($r['Продадени'] * $percent / 100)) continue;
    
            $key = $r['Група'] . '|' . $r['Стока'];
    
            if (!isset($aggregated[$key])) {
                $aggregated[$key] = [
                    'Група'      => $r['Група'],
                    'Стока'      => $r['Стока'],
                    'minDate'    => $d,
                    'maxDate'    => $d,
                    'Продадени'  => (int)$r['Продадени'],
                    'Налични'    => (int)$r['Налични'],
                ];
            } else {
                if ($d < $aggregated[$key]['minDate']) $aggregated[$key]['minDate'] = $d;
                if ($d > $aggregated[$key]['maxDate']) $aggregated[$key]['maxDate'] = $d;
                $aggregated[$key]['Продадени'] += (int)$r['Продадени'];
                $aggregated[$key]['Налични']   += (int)$r['Налични'];
            }
        }
    
        // === 3) Филтър за залежала стока ===
        $final = [];
    
        foreach ($aggregated as $r) {
            if ($r['maxDate'] < $staleBefore && $r['Налични'] > 0) {
                $final[] = [
                    'Група'             => $r['Група'],
                    'Стока'             => $r['Стока'],
                    'Последна доставка' => $r['maxDate']->format("d.m.Y"),
                    'Продадени'         => $r['Продадени'],
                    'Налични'           => $r['Налични'],
                ];
            }
        }
    
        // Ако няма залежали → върни всички
        if (empty($final)) {
            foreach ($aggregated as $r) {
                $final[] = [
                    'Група'             => $r['Група'],
                    'Стока'             => $r['Стока'],
                    'Последна доставка' => $r['maxDate']->format("d.m.Y"),
                    'Продадени'         => $r['Продадени'],
                    'Налични'           => $r['Налични'],
                ];
            }
        }
    
        // === 4) Сортиране ===
        usort($final, fn($a,$b)=> $a['Последна доставка'] <=> $b['Последна доставка']);
    

        $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        $sheet->setTitle("Залежала стока");
    
        // === HEADER TITLE ===
        $sheet->mergeCells("A1:E1");
        $sheet->setCellValue("A1", "АНАЛИЗ НА ЗАЛЕЖАЛА СТОКА");
        $sheet->getStyle("A1")->applyFromArray([
            'font'=>[
                'bold'=>true, 'size'=>18, 'color'=>['rgb'=>'FFFFFF']
            ],
            'alignment'=>['horizontal'=>'center'],
            'fill'=>[
                'fillType'=>'solid',
                'startColor'=>['rgb'=>'1E88E5']
            ]
        ]);
    
        // === SUB HEADER ===
        $sheet->mergeCells("A2:E2");
        $sheet->setCellValue("A2", "Период: {$monthsBack} месеца | Процент: {$percent}%");
        $sheet->getStyle("A2")->applyFromArray([
            'font'=>['size'=>11,'italic'=>true,'color'=>['rgb'=>'444444']],
            'alignment'=>['horizontal'=>'center']
        ]);
    
        // === TABLE HEADERS ===
        $headers = ["Група","Стока","Последна доставка","Продадени","Налични"];
        $sheet->fromArray($headers, null, "A3");
    
        $sheet->getStyle("A3:E3")->applyFromArray([
            'font'=>['bold'=>true,'size'=>12],
            'fill'=>['fillType'=>'solid','startColor'=>['rgb'=>'FCE4EC']],
            'borders'=>[
                'allBorders'=>['borderStyle'=>'thin','color'=>['rgb'=>'BBBBBB']]
            ],
            'alignment'=>['horizontal'=>'center']
        ]);
    
        // === TABLE ROWS ===
        $row = 4;
        $zebra = ["FFFFFF","F7F7F7"];  // редуващи се редове
        $i = 0;
    
        foreach ($final as $r) {
            $sheet->fromArray([
                $r['Група'],
                $r['Стока'],
                $r['Последна доставка'],
                $r['Продадени'],
                $r['Налични']
            ], null, "A{$row}");
    
            // 🟣 Zebra background
            $sheet->getStyle("A{$row}:E{$row}")->applyFromArray([
                'fill'=>[
                    'fillType'=>'solid',
                    'startColor'=>['rgb'=>$zebra[$i % 2]]
                ],
                'borders'=>[
                    'allBorders'=>['borderStyle'=>'thin','color'=>['rgb'=>'DDDDDD']]
                ]
            ]);
    
            $i++;
            $row++;
        }
    
        // === AUTO SIZE ===
        foreach (range('A','E') as $col)
            $sheet->getColumnDimension($col)->setAutoSize(true);
    
        // === EXPORT ===
        header("Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
        header("Content-Disposition: attachment; filename=\"zalejala_stoka.xlsx\"");
        header("Cache-Control: max-age=0");
    
        $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
        $writer->save("php://output");
        exit;
    }
    
    
    
    public function stockGroups()
    {
        $stockData = include APPPATH . 'Controllers/Dashboard/StockMovementData.php';
    
        $groups = [];
        $subgroups = [];
    
        foreach ($stockData as $row) {
            $g = trim($row['Главна група']);
            $s = trim($row['Подгрупа']);
    
            if ($g !== "" && !in_array($g, $groups)) {
                $groups[] = $g;
            }
    
            if (!isset($subgroups[$g])) {
                $subgroups[$g] = [];
            }
    
            if ($s !== "" && !in_array($s, $subgroups[$g])) {
                $subgroups[$g][] = $s;
            }
        }
    
        sort($groups);
        foreach ($subgroups as &$list) sort($list);
    
        return $this->response->setJSON([
            'groups'    => $groups,
            'subgroups' => $subgroups
        ]);
    }
    
    
    private function calculateStockMovement($stockData, $from, $to, $group, $sub)
    {
        $fromDate = $from ? \DateTime::createFromFormat('Y-m-d', $from) : null;
        $toDate   = $to ? \DateTime::createFromFormat('Y-m-d', $to) : null;
    
        // Филтри
        if ($fromDate && $toDate) {
            $stockData = array_filter($stockData, function($r) use ($fromDate, $toDate) {
                $d = \DateTime::createFromFormat('m/d/Y', $r['От дата']);
                return $d && $d >= $fromDate && $d <= $toDate;
            });
        }
    
        if ($group) {
            $stockData = array_filter($stockData, fn($r) => $r['Главна група'] === $group);
        }
    
        if ($sub) {
            $stockData = array_filter($stockData, fn($r) => $r['Подгрупа'] === $sub);
        }
    
        // Групиране като JS
        $products = [];
    
        foreach ($stockData as $r) {
            $key = $r['Стока'];
    
            $qty   = (float)$r['Количество'];
            $turn  = (float)$r['Оборот'];
            $profit = (float)$r['Чист приход'];
    
            // Цена 1 бр (без ДДС)
            $costPerUnit = $turn - $profit; 
            $totalCost   = $qty * $costPerUnit;
    
            if (!isset($products[$key])) {
                $products[$key] = [
                    'Главна група'        => $r['Главна група'],
                    'Подгрупа'            => $r['Подгрупа'],
                    'Стока'               => $r['Стока'],
                    'Наличност'           => 0,
                    'Доставна цена'       => $costPerUnit,
                    'ОБЩ_ТОТАЛ'           => 0,
                    'Продадени'           => 0,
                    'Оборот'              => 0,
                    'Чист приход'         => 0,
                    'Последна доставка'   => $r['От дата'],
                ];
            }
    
            $products[$key]['Наличност']  += $qty;
            $products[$key]['Продадени']  += $qty;
            $products[$key]['Оборот']     += $turn;
            $products[$key]['Чист приход']+= $profit;
            $products[$key]['ОБЩ_ТОТАЛ']  += $totalCost;
    
            // Последна доставка
            $cur = \DateTime::createFromFormat('m/d/Y', $r['От дата']);
            $last = \DateTime::createFromFormat('m/d/Y', $products[$key]['Последна доставка']);
    
            if ($cur > $last) {
                $products[$key]['Последна доставка'] = $r['От дата'];
            }
        }
    
        return array_values($products);
    }
    
    
    private function calculateBestSellers($stockData, $from, $to, $group, $sub)
    {
        $fromDate = $from ? \DateTime::createFromFormat('Y-m-d', $from) : null;
        $toDate   = $to ? \DateTime::createFromFormat('Y-m-d', $to) : null;
    
        // Филтри
        if ($fromDate && $toDate) {
            $stockData = array_filter($stockData, function($r) use ($fromDate, $toDate) {
                $d = \DateTime::createFromFormat('m/d/Y', $r['От дата']);
                return $d && $d >= $fromDate && $d <= $toDate;
            });
        }
    
        if ($group) {
            $stockData = array_filter($stockData, fn($r) => $r['Главна група'] === $group);
        }
    
        if ($sub) {
            $stockData = array_filter($stockData, fn($r) => $r['Подгрупа'] === $sub);
        }
    
        // Групиране
        $products = [];
    
        foreach ($stockData as $r) {
            $p = $r['Стока'];
    
            if (!isset($products[$p])) {
                $products[$p] = [
                    'Продукт'      => $p,
                    'Група'        => $r['Главна група'],
                    'Продадени'    => 0,
                    'Оборот'       => 0,
                    'Чист приход'  => 0,
                ];
            }
    
            $products[$p]['Продадени']   += (float)$r['Количество'];
            $products[$p]['Оборот']      += (float)$r['Оборот'];
            $products[$p]['Чист приход'] += (float)$r['Чист приход'];
        }
    
        // Сортиране
        usort($products, fn($a,$b)=>$b['Чист приход']<=>$a['Чист приход']);
    
        return array_slice($products, 0, 20);
    }


}
